import { useState } from 'react';
import { Eye, Search, Filter } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import type { Order } from '@/types';
import { toast } from 'sonner';

export function AdminOrders() {
  const { language, t, dir } = useLanguage();
  const { orders, updateOrderStatus } = useData();
  const [searchQuery, setSearchQuery] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [selectedOrder, setSelectedOrder] = useState<Order | null>(null);

  const filteredOrders = orders.filter((order) => {
    const matchesSearch =
      order.id.toLowerCase().includes(searchQuery.toLowerCase()) ||
      order.phone.includes(searchQuery);
    const matchesStatus = statusFilter === 'all' || order.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  const handleStatusChange = (orderId: string, newStatus: Order['status']) => {
    updateOrderStatus(orderId, newStatus);
    toast.success(t('وضعیت سفارش به‌روزرسانی شد', 'Order status updated'));
  };

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat(language === 'fa' ? 'fa-IR' : 'en-US').format(price);
  };

  const getStatusBadge = (status: Order['status']) => {
    const styles = {
      pending: 'bg-yellow-100 text-yellow-800',
      processing: 'bg-blue-100 text-blue-800',
      shipped: 'bg-purple-100 text-purple-800',
      delivered: 'bg-green-100 text-green-800',
      cancelled: 'bg-red-100 text-red-800',
    };

    const labels = {
      pending: t('در انتظار', 'Pending'),
      processing: t('در حال پردازش', 'Processing'),
      shipped: t('ارسال شده', 'Shipped'),
      delivered: t('تحویل داده شده', 'Delivered'),
      cancelled: t('لغو شده', 'Cancelled'),
    };

    return <span className={`px-2 py-1 rounded text-xs ${styles[status]}`}>{labels[status]}</span>;
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <h1 className="text-2xl md:text-3xl font-bold">{t('مدیریت سفارشات', 'Manage Orders')}</h1>
      </div>

      {/* Filters */}
      <div className="flex flex-col sm:flex-row gap-4">
        <div className="relative flex-1 max-w-md">
          <Search className="absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
          <Input
            type="search"
            placeholder={t('جستجو در سفارشات...', 'Search orders...')}
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pr-10"
            dir={dir}
          />
        </div>
        <Select value={statusFilter} onValueChange={setStatusFilter}>
          <SelectTrigger className="w-40">
            <Filter className="h-4 w-4 ml-2" />
            <SelectValue placeholder={t('وضعیت', 'Status')} />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">{t('همه', 'All')}</SelectItem>
            <SelectItem value="pending">{t('در انتظار', 'Pending')}</SelectItem>
            <SelectItem value="processing">{t('در حال پردازش', 'Processing')}</SelectItem>
            <SelectItem value="shipped">{t('ارسال شده', 'Shipped')}</SelectItem>
            <SelectItem value="delivered">{t('تحویل داده شده', 'Delivered')}</SelectItem>
            <SelectItem value="cancelled">{t('لغو شده', 'Cancelled')}</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* Orders Table */}
      <div className="bg-card rounded-lg border overflow-hidden">
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>{t('کد سفارش', 'Order ID')}</TableHead>
                <TableHead>{t('مشتری', 'Customer')}</TableHead>
                <TableHead>{t('مبلغ', 'Amount')}</TableHead>
                <TableHead>{t('وضعیت', 'Status')}</TableHead>
                <TableHead>{t('تاریخ', 'Date')}</TableHead>
                <TableHead className="text-left">{t('عملیات', 'Actions')}</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredOrders.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                    {t('سفارشی یافت نشد', 'No orders found')}
                  </TableCell>
                </TableRow>
              ) : (
                filteredOrders.map((order) => (
                  <TableRow key={order.id}>
                    <TableCell className="font-medium">#{order.id.slice(-6)}</TableCell>
                    <TableCell>
                      <div>
                        <p>{order.phone}</p>
                        <p className="text-xs text-muted-foreground line-clamp-1">{order.shippingAddress}</p>
                      </div>
                    </TableCell>
                    <TableCell>{formatPrice(order.totalAmount)}</TableCell>
                    <TableCell>{getStatusBadge(order.status)}</TableCell>
                    <TableCell className="text-sm text-muted-foreground">
                      {new Date(order.createdAt).toLocaleDateString(language === 'fa' ? 'fa-IR' : 'en-US')}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Select
                          value={order.status}
                          onValueChange={(value) => handleStatusChange(order.id, value as Order['status'])}
                        >
                          <SelectTrigger className="w-32">
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="pending">{t('در انتظار', 'Pending')}</SelectItem>
                            <SelectItem value="processing">{t('در حال پردازش', 'Processing')}</SelectItem>
                            <SelectItem value="shipped">{t('ارسال شده', 'Shipped')}</SelectItem>
                            <SelectItem value="delivered">{t('تحویل داده شده', 'Delivered')}</SelectItem>
                            <SelectItem value="cancelled">{t('لغو شده', 'Cancelled')}</SelectItem>
                          </SelectContent>
                        </Select>
                        <Button variant="ghost" size="icon" onClick={() => setSelectedOrder(order)}>
                          <Eye className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>

      {/* Order Details Dialog */}
      <Dialog open={!!selectedOrder} onOpenChange={() => setSelectedOrder(null)}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>
              {t('جزئیات سفارش', 'Order Details')} #{selectedOrder?.id.slice(-6)}
            </DialogTitle>
          </DialogHeader>
          {selectedOrder && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">{t('مشتری', 'Customer')}</p>
                  <p className="font-medium">{selectedOrder.phone}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">{t('تاریخ', 'Date')}</p>
                  <p className="font-medium">
                    {new Date(selectedOrder.createdAt).toLocaleString(language === 'fa' ? 'fa-IR' : 'en-US')}
                  </p>
                </div>
              </div>

              <div>
                <p className="text-sm text-muted-foreground">{t('آدرس', 'Address')}</p>
                <p>{selectedOrder.shippingAddress}</p>
              </div>

              {selectedOrder.notes && (
                <div>
                  <p className="text-sm text-muted-foreground">{t('توضیحات', 'Notes')}</p>
                  <p>{selectedOrder.notes}</p>
                </div>
              )}

              <div>
                <p className="text-sm text-muted-foreground mb-2">{t('محصولات', 'Products')}</p>
                <div className="border rounded-lg overflow-hidden">
                  <table className="w-full">
                    <thead className="bg-muted">
                      <tr>
                        <th className="text-right py-2 px-4">{t('محصول', 'Product')}</th>
                        <th className="text-right py-2 px-4">{t('تعداد', 'Qty')}</th>
                        <th className="text-right py-2 px-4">{t('قیمت', 'Price')}</th>
                      </tr>
                    </thead>
                    <tbody>
                      {selectedOrder.items.map(({ product, quantity }, index) => (
                        <tr key={index} className="border-t">
                          <td className="py-2 px-4">
                            {language === 'fa' ? product.name : product.nameEn}
                          </td>
                          <td className="py-2 px-4">{quantity}</td>
                          <td className="py-2 px-4">{formatPrice(product.price * quantity)}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </div>

              <div className="flex justify-between items-center pt-4 border-t">
                <p className="text-sm text-muted-foreground">{t('جمع کل', 'Total')}</p>
                <p className="text-xl font-bold text-primary">
                  {formatPrice(selectedOrder.totalAmount)} {t('تومان', 'Toman')}
                </p>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
