import { useState } from 'react';
import { Search, Shield, User } from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { useLanguage } from '@/contexts/LanguageContext';
import { useAuth } from '@/contexts/AuthContext';

interface UserData {
  id: string;
  name: string;
  email: string;
  phone?: string;
  isAdmin: boolean;
  createdAt: string;
}

export function AdminUsers() {
  const { language, t, dir } = useLanguage();
  const { user: currentUser } = useAuth();
  const [searchQuery, setSearchQuery] = useState('');

  // Get users from localStorage
  const users: UserData[] = (() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('users');
      if (saved) {
        return JSON.parse(saved).map((u: any) => ({
          id: u.id,
          name: u.name,
          email: u.email,
          phone: u.phone,
          isAdmin: u.isAdmin,
          createdAt: u.createdAt,
        }));
      }
    }
    return [];
  })();

  const filteredUsers = users.filter(
    (u) =>
      u.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
      u.email.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <h1 className="text-2xl md:text-3xl font-bold">{t('مدیریت کاربران', 'Manage Users')}</h1>
      </div>

      {/* Search */}
      <div className="relative max-w-md">
        <Search className="absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
        <Input
          type="search"
          placeholder={t('جستجو در کاربران...', 'Search users...')}
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="pr-10"
          dir={dir}
        />
      </div>

      {/* Users Table */}
      <div className="bg-card rounded-lg border overflow-hidden">
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>{t('کاربر', 'User')}</TableHead>
                <TableHead>{t('ایمیل', 'Email')}</TableHead>
                <TableHead>{t('نقش', 'Role')}</TableHead>
                <TableHead>{t('تاریخ ثبت‌نام', 'Register Date')}</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredUsers.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={4} className="text-center py-8 text-muted-foreground">
                    {t('کاربری یافت نشد', 'No users found')}
                  </TableCell>
                </TableRow>
              ) : (
                filteredUsers.map((user) => (
                  <TableRow key={user.id} className={user.id === currentUser?.id ? 'bg-primary/5' : ''}>
                    <TableCell>
                      <div className="flex items-center gap-3">
                        <div className="w-10 h-10 bg-primary/10 rounded-full flex items-center justify-center">
                          <User className="h-5 w-5 text-primary" />
                        </div>
                        <div>
                          <p className="font-medium">{user.name}</p>
                          {user.id === currentUser?.id && (
                            <Badge variant="secondary" className="text-xs">
                              {t('شما', 'You')}
                            </Badge>
                          )}
                        </div>
                      </div>
                    </TableCell>
                    <TableCell dir="ltr">{user.email}</TableCell>
                    <TableCell>
                      {user.isAdmin ? (
                        <span className="flex items-center gap-1 text-primary">
                          <Shield className="h-4 w-4" />
                          {t('مدیر', 'Admin')}
                        </span>
                      ) : (
                        <span className="flex items-center gap-1 text-muted-foreground">
                          <User className="h-4 w-4" />
                          {t('کاربر', 'User')}
                        </span>
                      )}
                    </TableCell>
                    <TableCell className="text-sm text-muted-foreground">
                      {new Date(user.createdAt).toLocaleDateString(
                        language === 'fa' ? 'fa-IR' : 'en-US'
                      )}
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-card rounded-lg border p-4">
          <p className="text-sm text-muted-foreground">{t('کل کاربران', 'Total Users')}</p>
          <p className="text-2xl font-bold">{users.length}</p>
        </div>
        <div className="bg-card rounded-lg border p-4">
          <p className="text-sm text-muted-foreground">{t('مدیران', 'Admins')}</p>
          <p className="text-2xl font-bold">{users.filter((u) => u.isAdmin).length}</p>
        </div>
        <div className="bg-card rounded-lg border p-4">
          <p className="text-sm text-muted-foreground">{t('کاربران عادی', 'Regular Users')}</p>
          <p className="text-2xl font-bold">{users.filter((u) => !u.isAdmin).length}</p>
        </div>
      </div>
    </div>
  );
}
