import { useState } from 'react';
import { Link, Outlet, useLocation, useNavigate } from 'react-router-dom';
import {
  LayoutDashboard,
  Package,
  Layers,
  FileText,
  Image,
  Settings,
  ShoppingCart,
  Users,
  Menu,
  X,
  LogOut,
  ChevronLeft,
  ChevronRight,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useLanguage } from '@/contexts/LanguageContext';
import { useAuth } from '@/contexts/AuthContext';
import { toast } from 'sonner';

const menuItems = [
  { path: '/admin', icon: LayoutDashboard, label: 'داشبورد', labelEn: 'Dashboard' },
  { path: '/admin/products', icon: Package, label: 'محصولات', labelEn: 'Products' },
  { path: '/admin/categories', icon: Layers, label: 'دسته‌بندی‌ها', labelEn: 'Categories' },
  { path: '/admin/orders', icon: ShoppingCart, label: 'سفارشات', labelEn: 'Orders' },
  { path: '/admin/blog', icon: FileText, label: 'وبلاگ', labelEn: 'Blog' },
  { path: '/admin/banners', icon: Image, label: 'بنرها', labelEn: 'Banners' },
  { path: '/admin/users', icon: Users, label: 'کاربران', labelEn: 'Users' },
  { path: '/admin/settings', icon: Settings, label: 'تنظیمات', labelEn: 'Settings' },
];

export function AdminLayout() {
  const { language, t, dir } = useLanguage();
  const { logout, isAdmin } = useAuth();
  const location = useLocation();
  const navigate = useNavigate();
  const [isSidebarOpen, setIsSidebarOpen] = useState(true);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);

  if (!isAdmin) {
    navigate('/');
    return null;
  }

  const handleLogout = () => {
    logout();
    navigate('/');
  };

  const SidebarContent = () => (
    <>
      {/* Logo */}
      <div className="p-4 border-b">
        <Link to="/admin" className="flex items-center gap-3">
          <div className="w-10 h-10 bg-primary rounded-lg flex items-center justify-center">
            <span className="text-primary-foreground font-bold text-xl">A</span>
          </div>
          <div>
            <h1 className="font-bold">{t('پنل مدیریت', 'Admin Panel')}</h1>
            <p className="text-xs text-muted-foreground">{t('فروشگاه دیجیتال', 'Digital Shop')}</p>
          </div>
        </Link>
      </div>

      {/* Menu */}
      <nav className="flex-1 p-4 overflow-y-auto">
        <ul className="space-y-1">
          {menuItems.map((item) => {
            const isActive = location.pathname === item.path;
            return (
              <li key={item.path}>
                <Link
                  to={item.path}
                  className={`flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${
                    isActive
                      ? 'bg-primary text-primary-foreground'
                      : 'hover:bg-muted'
                  }`}
                  onClick={() => setIsMobileMenuOpen(false)}
                >
                  <item.icon className="h-5 w-5" />
                  <span>{language === 'fa' ? item.label : item.labelEn}</span>
                </Link>
              </li>
            );
          })}
        </ul>
      </nav>

      {/* Footer */}
      <div className="p-4 border-t">
        <Link to="/">
          <Button variant="outline" className="w-full mb-2">
            {dir === 'rtl' ? <ChevronLeft className="h-4 w-4 ml-2" /> : <ChevronRight className="h-4 w-4 ml-2" />}
            {t('بازگشت به سایت', 'Back to Site')}
          </Button>
        </Link>
        <Button variant="ghost" className="w-full text-destructive" onClick={handleLogout}>
          <LogOut className="h-4 w-4 ml-2" />
          {t('خروج', 'Logout')}
        </Button>
      </div>
    </>
  );

  return (
    <div className="min-h-screen bg-muted/50">
      {/* Mobile Header */}
      <div className="lg:hidden bg-background border-b p-4 flex items-center justify-between sticky top-0 z-50">
        <Link to="/admin" className="flex items-center gap-2">
          <div className="w-8 h-8 bg-primary rounded-lg flex items-center justify-center">
            <span className="text-primary-foreground font-bold">A</span>
          </div>
          <span className="font-bold">{t('پنل مدیریت', 'Admin')}</span>
        </Link>
        <Button variant="ghost" size="icon" onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}>
          {isMobileMenuOpen ? <X className="h-5 w-5" /> : <Menu className="h-5 w-5" />}
        </Button>
      </div>

      {/* Mobile Menu */}
      {isMobileMenuOpen && (
        <div className="lg:hidden fixed inset-0 top-16 bg-background z-40">
          <SidebarContent />
        </div>
      )}

      <div className="flex">
        {/* Sidebar - Desktop */}
        <aside
          className={`hidden lg:flex flex-col bg-background border-l h-screen sticky top-0 transition-all duration-300 ${
            isSidebarOpen ? 'w-64' : 'w-20'
          }`}
        >
          {/* Logo */}
          <div className="p-4 border-b flex items-center justify-between">
            <Link to="/admin" className={`flex items-center gap-3 ${!isSidebarOpen && 'justify-center'}`}>
              <div className="w-10 h-10 bg-primary rounded-lg flex items-center justify-center flex-shrink-0">
                <span className="text-primary-foreground font-bold text-xl">A</span>
              </div>
              {isSidebarOpen && (
                <div>
                  <h1 className="font-bold">{t('پنل مدیریت', 'Admin')}</h1>
                </div>
              )}
            </Link>
            {isSidebarOpen && (
              <Button variant="ghost" size="icon" onClick={() => setIsSidebarOpen(false)}>
                {dir === 'rtl' ? <ChevronRight className="h-4 w-4" /> : <ChevronLeft className="h-4 w-4" />}
              </Button>
            )}
          </div>

          {/* Toggle Button - Collapsed */}
          {!isSidebarOpen && (
            <div className="p-4 border-b flex justify-center">
              <Button variant="ghost" size="icon" onClick={() => setIsSidebarOpen(true)}>
                {dir === 'rtl' ? <ChevronLeft className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
              </Button>
            </div>
          )}

          {/* Menu */}
          <nav className="flex-1 p-4 overflow-y-auto">
            <ul className="space-y-1">
              {menuItems.map((item) => {
                const isActive = location.pathname === item.path;
                return (
                  <li key={item.path}>
                    <Link
                      to={item.path}
                      className={`flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${
                        isActive
                          ? 'bg-primary text-primary-foreground'
                          : 'hover:bg-muted'
                      } ${!isSidebarOpen && 'justify-center px-2'}`}
                      title={language === 'fa' ? item.label : item.labelEn}
                    >
                      <item.icon className="h-5 w-5 flex-shrink-0" />
                      {isSidebarOpen && <span>{language === 'fa' ? item.label : item.labelEn}</span>}
                    </Link>
                  </li>
                );
              })}
            </ul>
          </nav>

          {/* Footer */}
          <div className="p-4 border-t">
            {isSidebarOpen ? (
              <>
                <Link to="/">
                  <Button variant="outline" className="w-full mb-2">
                    {dir === 'rtl' ? <ChevronLeft className="h-4 w-4 ml-2" /> : <ChevronRight className="h-4 w-4 ml-2" />}
                    {t('بازگشت به سایت', 'Back to Site')}
                  </Button>
                </Link>
                <Button variant="ghost" className="w-full text-destructive" onClick={handleLogout}>
                  <LogOut className="h-4 w-4 ml-2" />
                  {t('خروج', 'Logout')}
                </Button>
              </>
            ) : (
              <div className="flex flex-col items-center gap-2">
                <Link to="/">
                  <Button variant="ghost" size="icon">
                    {dir === 'rtl' ? <ChevronLeft className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                  </Button>
                </Link>
                <Button variant="ghost" size="icon" onClick={handleLogout} className="text-destructive">
                  <LogOut className="h-4 w-4" />
                </Button>
              </div>
            )}
          </div>
        </aside>

        {/* Main Content */}
        <main className="flex-1 p-4 md:p-8">
          <Outlet />
        </main>
      </div>
    </div>
  );
}
