import React, { createContext, useContext, useState, useEffect } from 'react';
import type { Product, Category, BlogPost, SiteSettings, Banner, Order } from '@/types';
import { sampleProducts, sampleCategories, sampleBlogPosts, sampleBanners } from '@/data/sampleData';

interface DataContextType {
  // Products
  products: Product[];
  addProduct: (product: Omit<Product, 'id' | 'createdAt' | 'updatedAt'>) => void;
  updateProduct: (id: string, product: Partial<Product>) => void;
  deleteProduct: (id: string) => void;
  getProductById: (id: string) => Product | undefined;
  getProductsByCategory: (categoryId: string) => Product[];
  getFeaturedProducts: () => Product[];
  getNewProducts: () => Product[];
  
  // Categories
  categories: Category[];
  addCategory: (category: Omit<Category, 'id'>) => void;
  updateCategory: (id: string, category: Partial<Category>) => void;
  deleteCategory: (id: string) => void;
  
  // Blog Posts
  blogPosts: BlogPost[];
  addBlogPost: (post: Omit<BlogPost, 'id' | 'createdAt' | 'updatedAt' | 'views'>) => void;
  updateBlogPost: (id: string, post: Partial<BlogPost>) => void;
  deleteBlogPost: (id: string) => void;
  getBlogPostById: (id: string) => BlogPost | undefined;
  
  // Banners
  banners: Banner[];
  addBanner: (banner: Omit<Banner, 'id'>) => void;
  updateBanner: (id: string, banner: Partial<Banner>) => void;
  deleteBanner: (id: string) => void;
  
  // Site Settings
  siteSettings: SiteSettings;
  updateSiteSettings: (settings: Partial<SiteSettings>) => void;
  
  // Orders
  orders: Order[];
  addOrder: (order: Omit<Order, 'id' | 'createdAt' | 'updatedAt'>) => void;
  updateOrderStatus: (id: string, status: Order['status']) => void;
  
  // Export/Import
  exportData: () => string;
  importData: (jsonData: string) => boolean;
}

const DataContext = createContext<DataContextType | undefined>(undefined);

const defaultSiteSettings: SiteSettings = {
  siteName: 'فروشگاه دیجیتال',
  siteNameEn: 'Digital Shop',
  logo: '/logo.png',
  favicon: '/favicon.ico',
  description: 'فروشگاه آنلاین کالاهای دیجیتال با بهترین قیمت',
  descriptionEn: 'Online digital products store with best prices',
  keywords: 'دیجیتال, لپ‌تاپ, موبایل, تبلت, هدفون',
  phone: '021-12345678',
  email: 'info@digitalshop.ir',
  address: 'تهران، خیابان انقلاب',
  socialLinks: {
    instagram: 'https://instagram.com/digitalshop',
    telegram: 'https://t.me/digitalshop',
    whatsapp: 'https://wa.me/989123456789',
  },
  footerText: 'تمامی حقوق محفوظ است',
  footerTextEn: 'All rights reserved',
  maintenanceMode: false,
};

export function DataProvider({ children }: { children: React.ReactNode }) {
  // Initialize state from localStorage or sample data
  const [products, setProducts] = useState<Product[]>(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('products');
      return saved ? JSON.parse(saved) : sampleProducts;
    }
    return sampleProducts;
  });

  const [categories, setCategories] = useState<Category[]>(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('categories');
      return saved ? JSON.parse(saved) : sampleCategories;
    }
    return sampleCategories;
  });

  const [blogPosts, setBlogPosts] = useState<BlogPost[]>(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('blogPosts');
      return saved ? JSON.parse(saved) : sampleBlogPosts;
    }
    return sampleBlogPosts;
  });

  const [banners, setBanners] = useState<Banner[]>(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('banners');
      return saved ? JSON.parse(saved) : sampleBanners;
    }
    return sampleBanners;
  });

  const [siteSettings, setSiteSettings] = useState<SiteSettings>(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('siteSettings');
      return saved ? JSON.parse(saved) : defaultSiteSettings;
    }
    return defaultSiteSettings;
  });

  const [orders, setOrders] = useState<Order[]>(() => {
    if (typeof window !== 'undefined') {
      const saved = localStorage.getItem('orders');
      return saved ? JSON.parse(saved) : [];
    }
    return [];
  });

  // Persist to localStorage
  useEffect(() => {
    localStorage.setItem('products', JSON.stringify(products));
  }, [products]);

  useEffect(() => {
    localStorage.setItem('categories', JSON.stringify(categories));
  }, [categories]);

  useEffect(() => {
    localStorage.setItem('blogPosts', JSON.stringify(blogPosts));
  }, [blogPosts]);

  useEffect(() => {
    localStorage.setItem('banners', JSON.stringify(banners));
  }, [banners]);

  useEffect(() => {
    localStorage.setItem('siteSettings', JSON.stringify(siteSettings));
  }, [siteSettings]);

  useEffect(() => {
    localStorage.setItem('orders', JSON.stringify(orders));
  }, [orders]);

  // Product functions
  const addProduct = (product: Omit<Product, 'id' | 'createdAt' | 'updatedAt'>) => {
    const newProduct: Product = {
      ...product,
      id: `prod-${Date.now()}`,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
    };
    setProducts((prev) => [newProduct, ...prev]);
  };

  const updateProduct = (id: string, product: Partial<Product>) => {
    setProducts((prev) =>
      prev.map((p) =>
        p.id === id ? { ...p, ...product, updatedAt: new Date().toISOString() } : p
      )
    );
  };

  const deleteProduct = (id: string) => {
    setProducts((prev) => prev.filter((p) => p.id !== id));
  };

  const getProductById = (id: string) => products.find((p) => p.id === id);
  const getProductsByCategory = (categoryId: string) =>
    products.filter((p) => p.category === categoryId);
  const getFeaturedProducts = () => products.filter((p) => p.isFeatured);
  const getNewProducts = () => products.filter((p) => p.isNew).slice(0, 8);

  // Category functions
  const addCategory = (category: Omit<Category, 'id'>) => {
    const newCategory: Category = {
      ...category,
      id: `cat-${Date.now()}`,
    };
    setCategories((prev) => [...prev, newCategory]);
  };

  const updateCategory = (id: string, category: Partial<Category>) => {
    setCategories((prev) =>
      prev.map((c) => (c.id === id ? { ...c, ...category } : c))
    );
  };

  const deleteCategory = (id: string) => {
    setCategories((prev) => prev.filter((c) => c.id !== id));
  };

  // Blog functions
  const addBlogPost = (post: Omit<BlogPost, 'id' | 'createdAt' | 'updatedAt' | 'views'>) => {
    const newPost: BlogPost = {
      ...post,
      id: `blog-${Date.now()}`,
      views: 0,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
    };
    setBlogPosts((prev) => [newPost, ...prev]);
  };

  const updateBlogPost = (id: string, post: Partial<BlogPost>) => {
    setBlogPosts((prev) =>
      prev.map((p) =>
        p.id === id ? { ...p, ...post, updatedAt: new Date().toISOString() } : p
      )
    );
  };

  const deleteBlogPost = (id: string) => {
    setBlogPosts((prev) => prev.filter((p) => p.id !== id));
  };

  const getBlogPostById = (id: string) => blogPosts.find((p) => p.id === id);

  // Banner functions
  const addBanner = (banner: Omit<Banner, 'id'>) => {
    const newBanner: Banner = {
      ...banner,
      id: `banner-${Date.now()}`,
    };
    setBanners((prev) => [...prev, newBanner]);
  };

  const updateBanner = (id: string, banner: Partial<Banner>) => {
    setBanners((prev) =>
      prev.map((b) => (b.id === id ? { ...b, ...banner } : b))
    );
  };

  const deleteBanner = (id: string) => {
    setBanners((prev) => prev.filter((b) => b.id !== id));
  };

  // Site settings functions
  const updateSiteSettings = (settings: Partial<SiteSettings>) => {
    setSiteSettings((prev) => ({ ...prev, ...settings }));
  };

  // Order functions
  const addOrder = (order: Omit<Order, 'id' | 'createdAt' | 'updatedAt'>) => {
    const newOrder: Order = {
      ...order,
      id: `order-${Date.now()}`,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
    };
    setOrders((prev) => [newOrder, ...prev]);
  };

  const updateOrderStatus = (id: string, status: Order['status']) => {
    setOrders((prev) =>
      prev.map((o) =>
        o.id === id ? { ...o, status, updatedAt: new Date().toISOString() } : o
      )
    );
  };

  // Export/Import functions
  const exportData = () => {
    const data = {
      products,
      categories,
      blogPosts,
      banners,
      siteSettings,
      orders,
      exportDate: new Date().toISOString(),
    };
    return JSON.stringify(data, null, 2);
  };

  const importData = (jsonData: string) => {
    try {
      const data = JSON.parse(jsonData);
      if (data.products) setProducts(data.products);
      if (data.categories) setCategories(data.categories);
      if (data.blogPosts) setBlogPosts(data.blogPosts);
      if (data.banners) setBanners(data.banners);
      if (data.siteSettings) setSiteSettings(data.siteSettings);
      if (data.orders) setOrders(data.orders);
      return true;
    } catch (error) {
      console.error('Import error:', error);
      return false;
    }
  };

  return (
    <DataContext.Provider
      value={{
        products,
        addProduct,
        updateProduct,
        deleteProduct,
        getProductById,
        getProductsByCategory,
        getFeaturedProducts,
        getNewProducts,
        categories,
        addCategory,
        updateCategory,
        deleteCategory,
        blogPosts,
        addBlogPost,
        updateBlogPost,
        deleteBlogPost,
        getBlogPostById,
        banners,
        addBanner,
        updateBanner,
        deleteBanner,
        siteSettings,
        updateSiteSettings,
        orders,
        addOrder,
        updateOrderStatus,
        exportData,
        importData,
      }}
    >
      {children}
    </DataContext.Provider>
  );
}

export function useData() {
  const context = useContext(DataContext);
  if (context === undefined) {
    throw new Error('useData must be used within a DataProvider');
  }
  return context;
}
