import { Link } from 'react-router-dom';
import { Calendar, User, Eye, ArrowLeft, ArrowRight } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';

export function Blog() {
  const { language, t, dir } = useLanguage();
  const { blogPosts } = useData();

  const publishedPosts = blogPosts.filter((p) => p.isPublished).sort((a, b) => 
    new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime()
  );

  const featuredPost = publishedPosts[0];
  const otherPosts = publishedPosts.slice(1);

  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString(language === 'fa' ? 'fa-IR' : 'en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  };

  return (
    <div className="min-h-screen py-8">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-3xl md:text-4xl font-bold mb-4">{t('وبلاگ', 'Blog')}</h1>
          <p className="text-muted-foreground max-w-2xl mx-auto">
            {t('جدیدترین اخبار، مقالات و راهنماهای دنیای تکنولوژی', 'Latest news, articles and guides from the tech world')}
          </p>
        </div>

        {/* Search */}
        <div className="max-w-md mx-auto mb-12">
          <div className="relative">
            <Input
              type="search"
              placeholder={t('جستجو در مقالات...', 'Search articles...')}
              className="w-full pr-10"
              dir={dir}
            />
            <button className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground">
              🔍
            </button>
          </div>
        </div>

        {/* Featured Post */}
        {featuredPost && (
          <div className="mb-12">
            <Link to={`/blog/${featuredPost.id}`}>
              <div className="relative rounded-2xl overflow-hidden group">
                <div className="aspect-[21/9] md:aspect-[3/1]">
                  <img
                    src={featuredPost.image}
                    alt={language === 'fa' ? featuredPost.title : featuredPost.titleEn}
                    className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/40 to-transparent" />
                </div>
                <div className="absolute bottom-0 left-0 right-0 p-6 md:p-8 text-white">
                  <Badge className="mb-3 bg-primary">{t('ویژه', 'Featured')}</Badge>
                  <h2 className="text-xl md:text-3xl font-bold mb-3">
                    {language === 'fa' ? featuredPost.title : featuredPost.titleEn}
                  </h2>
                  <p className="text-white/80 mb-4 line-clamp-2 max-w-2xl">
                    {language === 'fa' ? featuredPost.excerpt : featuredPost.excerptEn}
                  </p>
                  <div className="flex items-center gap-4 text-sm text-white/70">
                    <span className="flex items-center gap-1">
                      <User className="h-4 w-4" />
                      {featuredPost.author}
                    </span>
                    <span className="flex items-center gap-1">
                      <Calendar className="h-4 w-4" />
                      {formatDate(featuredPost.createdAt)}
                    </span>
                    <span className="flex items-center gap-1">
                      <Eye className="h-4 w-4" />
                      {featuredPost.views}
                    </span>
                  </div>
                </div>
              </div>
            </Link>
          </div>
        )}

        {/* Posts Grid */}
        {otherPosts.length > 0 && (
          <div>
            <h2 className="text-2xl font-bold mb-6">{t('مقالات اخیر', 'Recent Articles')}</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {otherPosts.map((post) => (
                <Link key={post.id} to={`/blog/${post.id}`}>
                  <article className="bg-card rounded-xl overflow-hidden border hover:shadow-lg transition-shadow h-full flex flex-col">
                    <div className="aspect-video overflow-hidden">
                      <img
                        src={post.image}
                        alt={language === 'fa' ? post.title : post.titleEn}
                        className="w-full h-full object-cover hover:scale-105 transition-transform duration-300"
                      />
                    </div>
                    <div className="p-5 flex-1 flex flex-col">
                      <div className="flex items-center gap-2 mb-3">
                        <Badge variant="secondary">{post.category}</Badge>
                      </div>
                      <h3 className="font-bold text-lg mb-2 line-clamp-2 hover:text-primary transition-colors">
                        {language === 'fa' ? post.title : post.titleEn}
                      </h3>
                      <p className="text-muted-foreground text-sm line-clamp-2 mb-4 flex-1">
                        {language === 'fa' ? post.excerpt : post.excerptEn}
                      </p>
                      <div className="flex items-center justify-between text-sm text-muted-foreground">
                        <span className="flex items-center gap-1">
                          <Calendar className="h-4 w-4" />
                          {formatDate(post.createdAt)}
                        </span>
                        <span className="flex items-center gap-1">
                          <Eye className="h-4 w-4" />
                          {post.views}
                        </span>
                      </div>
                    </div>
                  </article>
                </Link>
              ))}
            </div>
          </div>
        )}

        {/* Categories */}
        <div className="mt-16">
          <h2 className="text-2xl font-bold mb-6">{t('دسته‌بندی‌ها', 'Categories')}</h2>
          <div className="flex flex-wrap gap-3">
            {Array.from(new Set(publishedPosts.map((p) => p.category))).map((category) => (
              <Badge key={category} variant="outline" className="text-base px-4 py-2 cursor-pointer hover:bg-primary hover:text-primary-foreground transition-colors">
                {category}
              </Badge>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
}
