import { Link } from 'react-router-dom';
import { ArrowLeft, ArrowRight, Sparkles, TrendingUp, Package } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { HeroSlider } from '@/components/HeroSlider';
import { ProductCard } from '@/components/ProductCard';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';

export function Home() {
  const { language, t, dir } = useLanguage();
  const { categories, getFeaturedProducts, getNewProducts } = useData();

  const featuredProducts = getFeaturedProducts().slice(0, 8);
  const newProducts = getNewProducts().slice(0, 8);
  const activeCategories = categories.filter((c) => c.isActive).sort((a, b) => a.sortOrder - b.sortOrder);

  return (
    <div className="min-h-screen">
      {/* Hero Slider */}
      <HeroSlider />

      {/* Categories Section */}
      <section className="py-12 md:py-16">
        <div className="container mx-auto px-4">
          <div className="flex items-center justify-between mb-8">
            <h2 className="text-2xl md:text-3xl font-bold">{t('دسته‌بندی‌ها', 'Categories')}</h2>
            <Link to="/products">
              <Button variant="ghost" className="gap-2">
                {t('مشاهده همه', 'View All')}
                {dir === 'rtl' ? <ArrowLeft className="h-4 w-4" /> : <ArrowRight className="h-4 w-4" />}
              </Button>
            </Link>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-8 gap-4">
            {activeCategories.map((category) => (
              <Link
                key={category.id}
                to={`/category/${category.id}`}
                className="group"
              >
                <div className="aspect-square rounded-xl overflow-hidden bg-muted mb-3">
                  <img
                    src={category.image}
                    alt={language === 'fa' ? category.name : category.nameEn}
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                    loading="lazy"
                  />
                </div>
                <p className="text-center text-sm font-medium group-hover:text-primary transition-colors">
                  {language === 'fa' ? category.name : category.nameEn}
                </p>
              </Link>
            ))}
          </div>
        </div>
      </section>

      {/* Featured Products */}
      <section className="py-12 md:py-16 bg-muted/50">
        <div className="container mx-auto px-4">
          <div className="flex items-center justify-between mb-8">
            <div className="flex items-center gap-3">
              <Sparkles className="h-6 w-6 text-primary" />
              <h2 className="text-2xl md:text-3xl font-bold">{t('محصولات ویژه', 'Featured Products')}</h2>
            </div>
            <Link to="/products?featured=true">
              <Button variant="ghost" className="gap-2">
                {t('مشاهده همه', 'View All')}
                {dir === 'rtl' ? <ArrowLeft className="h-4 w-4" /> : <ArrowRight className="h-4 w-4" />}
              </Button>
            </Link>
          </div>

          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
            {featuredProducts.map((product) => (
              <ProductCard key={product.id} product={product} />
            ))}
          </div>
        </div>
      </section>

      {/* New Products */}
      <section className="py-12 md:py-16">
        <div className="container mx-auto px-4">
          <div className="flex items-center justify-between mb-8">
            <div className="flex items-center gap-3">
              <Package className="h-6 w-6 text-green-500" />
              <h2 className="text-2xl md:text-3xl font-bold">{t('جدیدترین محصولات', 'New Arrivals')}</h2>
            </div>
            <Link to="/products?new=true">
              <Button variant="ghost" className="gap-2">
                {t('مشاهده همه', 'View All')}
                {dir === 'rtl' ? <ArrowLeft className="h-4 w-4" /> : <ArrowRight className="h-4 w-4" />}
              </Button>
            </Link>
          </div>

          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
            {newProducts.map((product) => (
              <ProductCard key={product.id} product={product} />
            ))}
          </div>
        </div>
      </section>

      {/* Promotional Banner */}
      <section className="py-12 md:py-16">
        <div className="container mx-auto px-4">
          <div className="relative rounded-2xl overflow-hidden">
            <div
              className="absolute inset-0 bg-cover bg-center"
              style={{
                backgroundImage: 'url(https://images.unsplash.com/photo-1550009158-9ebf690569ba?w=1200)',
              }}
            >
              <div className="absolute inset-0 bg-gradient-to-r from-primary/90 to-primary/50" />
            </div>
            <div className="relative py-16 px-8 md:px-16 text-white">
              <div className="max-w-lg">
                <div className="flex items-center gap-2 mb-4">
                  <TrendingUp className="h-6 w-6" />
                  <span className="font-semibold">{t('پیشنهاد ویژه', 'Special Offer')}</span>
                </div>
                <h2 className="text-3xl md:text-4xl font-bold mb-4">
                  {t('تا 50% تخفیف روی لوازم جانبی', 'Up to 50% off on accessories')}
                </h2>
                <p className="text-white/90 mb-6">
                  {t('فرصت را از دست ندهید! تخفیف‌های ویژه روی هدفون، کیبورد و سایر لوازم جانبی',
                    'Don\'t miss out! Special discounts on headphones, keyboards and other accessories')}
                </p>
                <Link to="/category/cat-8">
                  <Button size="lg" variant="secondary">
                    {t('مشاهده محصولات', 'View Products')}
                  </Button>
                </Link>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Features */}
      <section className="py-12 md:py-16 bg-muted/50">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center p-6">
              <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                <Package className="h-8 w-8 text-primary" />
              </div>
              <h3 className="font-bold text-lg mb-2">{t('تحویل سریع', 'Fast Delivery')}</h3>
              <p className="text-muted-foreground">
                {t('ارسال به سراسر کشور در کمترین زمان ممکن', 'Nationwide delivery in minimum time')}
              </p>
            </div>
            <div className="text-center p-6">
              <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                <Sparkles className="h-8 w-8 text-primary" />
              </div>
              <h3 className="font-bold text-lg mb-2">{t('ضمانت اصالت', 'Authenticity Guarantee')}</h3>
              <p className="text-muted-foreground">
                {t('تمامی محصولات 100% اورجینال و با گارانتی', 'All products are 100% original with warranty')}
              </p>
            </div>
            <div className="text-center p-6">
              <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                <TrendingUp className="h-8 w-8 text-primary" />
              </div>
              <h3 className="font-bold text-lg mb-2">{t('قیمت مناسب', 'Best Prices')}</h3>
              <p className="text-muted-foreground">
                {t('بهترین قیمت‌ها در بازار با تضمین کیفیت', 'Best market prices with quality guarantee')}
              </p>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
}
