import { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import {
  Search,
  ShoppingCart,
  User,
  Menu,
  X,
  Sun,
  Moon,
  Globe,
  LogOut,
  LayoutDashboard,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
  DropdownMenuSeparator,
} from '@/components/ui/dropdown-menu';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { useLanguage } from '@/contexts/LanguageContext';
import { useTheme } from '@/contexts/ThemeContext';
import { useCart } from '@/contexts/CartContext';
import { useAuth } from '@/contexts/AuthContext';
import { useData } from '@/contexts/DataContext';

export function Header() {
  const { language, setLanguage, t, dir } = useLanguage();
  const { theme, toggleTheme } = useTheme();
  const { totalItems } = useCart();
  const { isAuthenticated, isAdmin, user, logout } = useAuth();
  const { siteSettings, categories } = useData();
  const navigate = useNavigate();
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      navigate(`/products?search=${encodeURIComponent(searchQuery)}`);
      setSearchQuery('');
    }
  };

  const activeCategories = categories.filter((c) => c.isActive).sort((a, b) => a.sortOrder - b.sortOrder);

  return (
    <header className="sticky top-0 z-50 w-full border-b bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
      {/* Top Bar */}
      <div className="bg-primary text-primary-foreground py-2">
        <div className="container mx-auto px-4 flex justify-between items-center text-sm">
          <div className="flex items-center gap-4">
            <span className="hidden sm:inline">{siteSettings.phone}</span>
            <span className="hidden sm:inline">|</span>
            <span className="hidden md:inline">{siteSettings.email}</span>
          </div>
          <div className="flex items-center gap-2">
            <span className="text-xs">{t('ارسال رایگان برای خرید بالای 5 میلیون', 'Free shipping over 50M')}</span>
          </div>
        </div>
      </div>

      {/* Main Header */}
      <div className="container mx-auto px-4 py-4">
        <div className="flex items-center justify-between gap-4">
          {/* Logo */}
          <Link to="/" className="flex items-center gap-2 flex-shrink-0">
            <div className="w-10 h-10 bg-primary rounded-lg flex items-center justify-center">
              <span className="text-primary-foreground font-bold text-xl">D</span>
            </div>
            <span className="font-bold text-xl hidden sm:block">
              {language === 'fa' ? siteSettings.siteName : siteSettings.siteNameEn}
            </span>
          </Link>

          {/* Search Bar - Desktop */}
          <form onSubmit={handleSearch} className="hidden md:flex flex-1 max-w-xl mx-4">
            <div className="relative w-full">
              <Input
                type="search"
                placeholder={t('جستجو در محصولات...', 'Search products...')}
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="w-full pr-10"
                dir={dir}
              />
              <button
                type="submit"
                className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
              >
                <Search className="h-4 w-4" />
              </button>
            </div>
          </form>

          {/* Actions */}
          <div className="flex items-center gap-2">
            {/* Theme Toggle */}
            <Button
              variant="ghost"
              size="icon"
              onClick={toggleTheme}
              className="hidden sm:flex"
            >
              {theme === 'light' ? <Moon className="h-5 w-5" /> : <Sun className="h-5 w-5" />}
            </Button>

            {/* Language Toggle */}
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setLanguage(language === 'fa' ? 'en' : 'fa')}
              className="hidden sm:flex"
            >
              <Globe className="h-5 w-5" />
            </Button>

            {/* Cart */}
            <Link to="/cart">
              <Button variant="ghost" size="icon" className="relative">
                <ShoppingCart className="h-5 w-5" />
                {totalItems > 0 && (
                  <Badge
                    variant="destructive"
                    className="absolute -top-1 -right-1 h-5 w-5 flex items-center justify-center p-0 text-xs"
                  >
                    {totalItems}
                  </Badge>
                )}
              </Button>
            </Link>

            {/* User Menu */}
            {isAuthenticated ? (
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="ghost" size="icon">
                    <User className="h-5 w-5" />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end" className="w-48">
                  <div className="px-3 py-2 text-sm font-medium">{user?.name}</div>
                  <DropdownMenuSeparator />
                  <DropdownMenuItem onClick={() => navigate('/profile')}>
                    <User className="h-4 w-4 ml-2" />
                    {t('پروفایل', 'Profile')}
                  </DropdownMenuItem>
                  {isAdmin && (
                    <DropdownMenuItem onClick={() => navigate('/admin')}>
                      <LayoutDashboard className="h-4 w-4 ml-2" />
                      {t('پنل مدیریت', 'Admin Panel')}
                    </DropdownMenuItem>
                  )}
                  <DropdownMenuSeparator />
                  <DropdownMenuItem onClick={logout} className="text-destructive">
                    <LogOut className="h-4 w-4 ml-2" />
                    {t('خروج', 'Logout')}
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            ) : (
              <Link to="/login">
                <Button variant="ghost" size="icon">
                  <User className="h-5 w-5" />
                </Button>
              </Link>
            )}

            {/* Mobile Menu Toggle */}
            <Button
              variant="ghost"
              size="icon"
              className="md:hidden"
              onClick={() => setIsMenuOpen(!isMenuOpen)}
            >
              {isMenuOpen ? <X className="h-5 w-5" /> : <Menu className="h-5 w-5" />}
            </Button>
          </div>
        </div>
      </div>

      {/* Navigation - Desktop */}
      <nav className="hidden md:block border-t">
        <div className="container mx-auto px-4">
          <ul className="flex items-center gap-6 py-3 overflow-x-auto">
            <li>
              <Link to="/" className="text-sm font-medium hover:text-primary transition-colors">
                {t('صفحه اصلی', 'Home')}
              </Link>
            </li>
            <li>
              <Link to="/products" className="text-sm font-medium hover:text-primary transition-colors">
                {t('همه محصولات', 'All Products')}
              </Link>
            </li>
            {activeCategories.slice(0, 6).map((category) => (
              <li key={category.id}>
                <Link
                  to={`/category/${category.id}`}
                  className="text-sm font-medium hover:text-primary transition-colors whitespace-nowrap"
                >
                  {language === 'fa' ? category.name : category.nameEn}
                </Link>
              </li>
            ))}
            <li>
              <Link to="/blog" className="text-sm font-medium hover:text-primary transition-colors">
                {t('وبلاگ', 'Blog')}
              </Link>
            </li>
            <li>
              <Link to="/contact" className="text-sm font-medium hover:text-primary transition-colors">
                {t('تماس با ما', 'Contact')}
              </Link>
            </li>
          </ul>
        </div>
      </nav>

      {/* Mobile Menu */}
      {isMenuOpen && (
        <div className="md:hidden border-t bg-background">
          <div className="container mx-auto px-4 py-4">
            {/* Mobile Search */}
            <form onSubmit={handleSearch} className="mb-4">
              <div className="relative">
                <Input
                  type="search"
                  placeholder={t('جستجو...', 'Search...')}
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="w-full pr-10"
                  dir={dir}
                />
                <button
                  type="submit"
                  className="absolute right-3 top-1/2 -translate-y-1/2"
                >
                  <Search className="h-4 w-4" />
                </button>
              </div>
            </form>

            {/* Mobile Navigation */}
            <ul className="space-y-2">
              <li>
                <Link
                  to="/"
                  className="block py-2 text-sm font-medium"
                  onClick={() => setIsMenuOpen(false)}
                >
                  {t('صفحه اصلی', 'Home')}
                </Link>
              </li>
              <li>
                <Link
                  to="/products"
                  className="block py-2 text-sm font-medium"
                  onClick={() => setIsMenuOpen(false)}
                >
                  {t('همه محصولات', 'All Products')}
                </Link>
              </li>
              {activeCategories.map((category) => (
                <li key={category.id}>
                  <Link
                    to={`/category/${category.id}`}
                    className="block py-2 text-sm font-medium"
                    onClick={() => setIsMenuOpen(false)}
                  >
                    {language === 'fa' ? category.name : category.nameEn}
                  </Link>
                </li>
              ))}
              <li>
                <Link
                  to="/blog"
                  className="block py-2 text-sm font-medium"
                  onClick={() => setIsMenuOpen(false)}
                >
                  {t('وبلاگ', 'Blog')}
                </Link>
              </li>
              <li>
                <Link
                  to="/contact"
                  className="block py-2 text-sm font-medium"
                  onClick={() => setIsMenuOpen(false)}
                >
                  {t('تماس با ما', 'Contact')}
                </Link>
              </li>
            </ul>

            {/* Mobile Actions */}
            <div className="flex items-center gap-4 mt-4 pt-4 border-t">
              <Button variant="outline" size="sm" onClick={toggleTheme} className="flex-1">
                {theme === 'light' ? <Moon className="h-4 w-4 ml-2" /> : <Sun className="h-4 w-4 ml-2" />}
                {theme === 'light' ? t('حالت تاریک', 'Dark Mode') : t('حالت روشن', 'Light Mode')}
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setLanguage(language === 'fa' ? 'en' : 'fa')}
                className="flex-1"
              >
                <Globe className="h-4 w-4 ml-2" />
                {language === 'fa' ? 'English' : 'فارسی'}
              </Button>
            </div>
          </div>
        </div>
      )}
    </header>
  );
}
