import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { CreditCard, Truck, MapPin, Phone, User, Check } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { useLanguage } from '@/contexts/LanguageContext';
import { useCart } from '@/contexts/CartContext';
import { useAuth } from '@/contexts/AuthContext';
import { useData } from '@/contexts/DataContext';
import { toast } from 'sonner';

export function Checkout() {
  const { language, t, dir } = useLanguage();
  const { items, totalPrice, clearCart } = useCart();
  const { user } = useAuth();
  const { addOrder } = useData();
  const navigate = useNavigate();

  const [formData, setFormData] = useState({
    name: user?.name || '',
    phone: user?.phone || '',
    address: user?.address || '',
    notes: '',
  });
  const [paymentMethod, setPaymentMethod] = useState('cod');
  const [isSubmitting, setIsSubmitting] = useState(false);

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat(language === 'fa' ? 'fa-IR' : 'en-US').format(price);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!formData.name || !formData.phone || !formData.address) {
      toast.error(t('لطفاً همه فیلدها را پر کنید', 'Please fill all fields'));
      return;
    }

    setIsSubmitting(true);

    // Create order
    const order = {
      userId: user?.id || 'guest',
      items: items.map(({ product, quantity }) => ({
        product: {
          ...product,
          createdAt: product.createdAt,
          updatedAt: product.updatedAt,
        },
        quantity,
      })),
      totalAmount: totalPrice,
      status: 'pending' as const,
      paymentStatus: 'pending' as const,
      shippingAddress: formData.address,
      phone: formData.phone,
      notes: formData.notes,
    };

    addOrder(order);

    // Update product stock
    items.forEach(({ product, quantity }) => {
      const updatedStock = product.stock - quantity;
      // This would be handled by the data context in a real app
    });

    toast.success(t('سفارش با موفقیت ثبت شد', 'Order placed successfully'));
    clearCart();
    navigate('/order-success');
    setIsSubmitting(false);
  };

  if (items.length === 0) {
    navigate('/cart');
    return null;
  }

  return (
    <div className="min-h-screen py-8">
      <div className="container mx-auto px-4">
        <h1 className="text-2xl md:text-3xl font-bold mb-8">{t('تکمیل خرید', 'Checkout')}</h1>

        <form onSubmit={handleSubmit}>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            {/* Form */}
            <div className="lg:col-span-2 space-y-6">
              {/* Shipping Info */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Truck className="h-5 w-5" />
                    {t('اطلاعات ارسال', 'Shipping Information')}
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="name">
                        <User className="h-4 w-4 inline ml-2" />
                        {t('نام و نام خانوادگی', 'Full Name')}
                      </Label>
                      <Input
                        id="name"
                        value={formData.name}
                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                        placeholder={t('نام خود را وارد کنید', 'Enter your name')}
                        dir={dir}
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="phone">
                        <Phone className="h-4 w-4 inline ml-2" />
                        {t('شماره موبایل', 'Phone Number')}
                      </Label>
                      <Input
                        id="phone"
                        type="tel"
                        value={formData.phone}
                        onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                        placeholder={t('مثال: 09123456789', 'e.g., 09123456789')}
                        dir="ltr"
                        required
                      />
                    </div>
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="address">
                      <MapPin className="h-4 w-4 inline ml-2" />
                      {t('آدرس کامل', 'Full Address')}
                    </Label>
                    <textarea
                      id="address"
                      value={formData.address}
                      onChange={(e) => setFormData({ ...formData, address: e.target.value })}
                      placeholder={t('آدرس دقیق خود را وارد کنید', 'Enter your exact address')}
                      className="w-full px-3 py-2 border rounded-md min-h-[100px]"
                      dir={dir}
                      required
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="notes">{t('توضیحات اضافی (اختیاری)', 'Additional Notes (Optional)')}</Label>
                    <textarea
                      id="notes"
                      value={formData.notes}
                      onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                      placeholder={t('توضیحات خاص برای سفارش', 'Special instructions for order')}
                      className="w-full px-3 py-2 border rounded-md"
                      dir={dir}
                    />
                  </div>
                </CardContent>
              </Card>

              {/* Payment Method */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <CreditCard className="h-5 w-5" />
                    {t('روش پرداخت', 'Payment Method')}
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <RadioGroup value={paymentMethod} onValueChange={setPaymentMethod} className="space-y-3">
                    <div className="flex items-center space-x-2 space-x-reverse p-4 border rounded-lg cursor-pointer hover:bg-muted transition-colors">
                      <RadioGroupItem value="cod" id="cod" />
                      <Label htmlFor="cod" className="flex-1 cursor-pointer">
                        <div className="font-medium">{t('پرداخت در محل', 'Cash on Delivery')}</div>
                        <div className="text-sm text-muted-foreground">
                          {t('پرداخت هنگام تحویل سفارش', 'Pay when you receive your order')}
                        </div>
                      </Label>
                    </div>

                    <div className="flex items-center space-x-2 space-x-reverse p-4 border rounded-lg cursor-pointer hover:bg-muted transition-colors opacity-50">
                      <RadioGroupItem value="online" id="online" disabled />
                      <Label htmlFor="online" className="flex-1 cursor-not-allowed">
                        <div className="font-medium">{t('پرداخت آنلاین', 'Online Payment')}</div>
                        <div className="text-sm text-muted-foreground">
                          {t('به زودی فعال می‌شود', 'Coming soon')}
                        </div>
                      </Label>
                    </div>

                    <div className="flex items-center space-x-2 space-x-reverse p-4 border rounded-lg cursor-pointer hover:bg-muted transition-colors opacity-50">
                      <RadioGroupItem value="card" id="card" disabled />
                      <Label htmlFor="card" className="flex-1 cursor-not-allowed">
                        <div className="font-medium">{t('کارت به کارت', 'Card Transfer')}</div>
                        <div className="text-sm text-muted-foreground">
                          {t('به زودی فعال می‌شود', 'Coming soon')}
                        </div>
                      </Label>
                    </div>
                  </RadioGroup>
                </CardContent>
              </Card>
            </div>

            {/* Order Summary */}
            <div className="lg:col-span-1">
              <Card className="sticky top-24">
                <CardHeader>
                  <CardTitle>{t('خلاصه سفارش', 'Order Summary')}</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  {/* Items */}
                  <div className="space-y-3 max-h-60 overflow-y-auto">
                    {items.map(({ product, quantity }) => (
                      <div key={product.id} className="flex gap-3">
                        <div className="w-16 h-16 bg-muted rounded overflow-hidden flex-shrink-0">
                          <img
                            src={product.images[0]}
                            alt={language === 'fa' ? product.name : product.nameEn}
                            className="w-full h-full object-cover"
                          />
                        </div>
                        <div className="flex-1 min-w-0">
                          <p className="text-sm font-medium line-clamp-1">
                            {language === 'fa' ? product.name : product.nameEn}
                          </p>
                          <p className="text-xs text-muted-foreground">
                            {quantity} × {formatPrice(product.price)}
                          </p>
                        </div>
                      </div>
                    ))}
                  </div>

                  <Separator />

                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">{t('جمع محصولات', 'Subtotal')}</span>
                      <span>{formatPrice(totalPrice)} {t('تومان', 'Toman')}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">{t('هزینه ارسال', 'Shipping')}</span>
                      <span className="text-green-600">{t('رایگان', 'Free')}</span>
                    </div>
                  </div>

                  <Separator />

                  <div className="flex justify-between text-lg font-bold">
                    <span>{t('جمع کل', 'Total')}</span>
                    <span className="text-primary">
                      {formatPrice(totalPrice)} {t('تومان', 'Toman')}
                    </span>
                  </div>

                  <Button type="submit" className="w-full" size="lg" disabled={isSubmitting}>
                    {isSubmitting ? (
                      <span className="flex items-center gap-2">
                        <span className="animate-spin">⏳</span>
                        {t('در حال پردازش...', 'Processing...')}
                      </span>
                    ) : (
                      <span className="flex items-center gap-2">
                        <Check className="h-5 w-5" />
                        {t('ثبت سفارش', 'Place Order')}
                      </span>
                    )}
                  </Button>
                </CardContent>
              </Card>
            </div>
          </div>
        </form>
      </div>
    </div>
  );
}
