import { useMemo } from 'react';
import {
  Package,
  ShoppingCart,
  Users,
  FileText,
  TrendingUp,
  TrendingDown,
  DollarSign,
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';

export function AdminDashboard() {
  const { language, t } = useLanguage();
  const { products, orders, blogPosts, categories } = useData();

  const stats = useMemo(() => {
    const totalRevenue = orders.reduce((sum, order) => sum + order.totalAmount, 0);
    const pendingOrders = orders.filter((o) => o.status === 'pending').length;
    const lowStockProducts = products.filter((p) => p.stock < 5).length;

    return {
      totalProducts: products.length,
      totalOrders: orders.length,
      totalRevenue,
      pendingOrders,
      totalBlogPosts: blogPosts.length,
      totalCategories: categories.length,
      lowStockProducts,
    };
  }, [products, orders, blogPosts, categories]);

  const recentOrders = orders.slice(0, 5);

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat(language === 'fa' ? 'fa-IR' : 'en-US').format(price);
  };

  const statCards = [
    {
      title: t('کل محصولات', 'Total Products'),
      value: stats.totalProducts,
      icon: Package,
      color: 'text-blue-600',
      bgColor: 'bg-blue-100 dark:bg-blue-900/30',
    },
    {
      title: t('کل سفارشات', 'Total Orders'),
      value: stats.totalOrders,
      icon: ShoppingCart,
      color: 'text-green-600',
      bgColor: 'bg-green-100 dark:bg-green-900/30',
    },
    {
      title: t('درآمد کل', 'Total Revenue'),
      value: `${formatPrice(stats.totalRevenue)} ${t('تومان', 'Toman')}`,
      icon: DollarSign,
      color: 'text-purple-600',
      bgColor: 'bg-purple-100 dark:bg-purple-900/30',
    },
    {
      title: t('سفارشات در انتظار', 'Pending Orders'),
      value: stats.pendingOrders,
      icon: TrendingUp,
      color: 'text-orange-600',
      bgColor: 'bg-orange-100 dark:bg-orange-900/30',
    },
    {
      title: t('مقالات وبلاگ', 'Blog Posts'),
      value: stats.totalBlogPosts,
      icon: FileText,
      color: 'text-pink-600',
      bgColor: 'bg-pink-100 dark:bg-pink-900/30',
    },
    {
      title: t('دسته‌بندی‌ها', 'Categories'),
      value: stats.totalCategories,
      icon: Package,
      color: 'text-cyan-600',
      bgColor: 'bg-cyan-100 dark:bg-cyan-900/30',
    },
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl md:text-3xl font-bold">{t('داشبورد', 'Dashboard')}</h1>
        <p className="text-muted-foreground">
          {new Date().toLocaleDateString(language === 'fa' ? 'fa-IR' : 'en-US', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric',
          })}
        </p>
      </div>

      {/* Stats Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {statCards.map((stat, index) => (
          <Card key={index}>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground mb-1">{stat.title}</p>
                  <p className="text-2xl font-bold">{stat.value}</p>
                </div>
                <div className={`w-12 h-12 ${stat.bgColor} rounded-full flex items-center justify-center`}>
                  <stat.icon className={`h-6 w-6 ${stat.color}`} />
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Alerts */}
      {stats.lowStockProducts > 0 && (
        <div className="bg-yellow-100 dark:bg-yellow-900/30 border border-yellow-300 dark:border-yellow-700 rounded-lg p-4 flex items-center gap-3">
          <TrendingDown className="h-5 w-5 text-yellow-600" />
          <p>
            <span className="font-bold">{t('هشدار:', 'Warning:')}</span>{' '}
            {t(
              `${stats.lowStockProducts} محصول موجودی کم دارند.`,
              `${stats.lowStockProducts} products have low stock.`
            )}
          </p>
        </div>
      )}

      {/* Recent Orders */}
      <Card>
        <CardHeader>
          <CardTitle>{t('سفارشات اخیر', 'Recent Orders')}</CardTitle>
        </CardHeader>
        <CardContent>
          {recentOrders.length === 0 ? (
            <p className="text-muted-foreground text-center py-8">
              {t('سفارشی یافت نشد', 'No orders found')}
            </p>
          ) : (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b">
                    <th className="text-right py-3 px-4">{t('کد سفارش', 'Order ID')}</th>
                    <th className="text-right py-3 px-4">{t('مشتری', 'Customer')}</th>
                    <th className="text-right py-3 px-4">{t('مبلغ', 'Amount')}</th>
                    <th className="text-right py-3 px-4">{t('وضعیت', 'Status')}</th>
                    <th className="text-right py-3 px-4">{t('تاریخ', 'Date')}</th>
                  </tr>
                </thead>
                <tbody>
                  {recentOrders.map((order) => (
                    <tr key={order.id} className="border-b hover:bg-muted/50">
                      <td className="py-3 px-4 font-medium">#{order.id.slice(-6)}</td>
                      <td className="py-3 px-4">{order.phone}</td>
                      <td className="py-3 px-4">{formatPrice(order.totalAmount)}</td>
                      <td className="py-3 px-4">
                        <span
                          className={`px-2 py-1 rounded text-xs ${
                            order.status === 'pending'
                              ? 'bg-yellow-100 text-yellow-800'
                              : order.status === 'processing'
                              ? 'bg-blue-100 text-blue-800'
                              : order.status === 'shipped'
                              ? 'bg-purple-100 text-purple-800'
                              : order.status === 'delivered'
                              ? 'bg-green-100 text-green-800'
                              : 'bg-red-100 text-red-800'
                          }`}
                        >
                          {order.status === 'pending' && t('در انتظار', 'Pending')}
                          {order.status === 'processing' && t('در حال پردازش', 'Processing')}
                          {order.status === 'shipped' && t('ارسال شده', 'Shipped')}
                          {order.status === 'delivered' && t('تحویل داده شده', 'Delivered')}
                          {order.status === 'cancelled' && t('لغو شده', 'Cancelled')}
                        </span>
                      </td>
                      <td className="py-3 px-4 text-sm text-muted-foreground">
                        {new Date(order.createdAt).toLocaleDateString(
                          language === 'fa' ? 'fa-IR' : 'en-US'
                        )}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
