import { useState, useRef } from 'react';
import { Save, Download, Upload, Globe, Instagram, Send, Phone } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { useLanguage } from '@/contexts/LanguageContext';
import { useData } from '@/contexts/DataContext';
import { toast } from 'sonner';

export function AdminSettings() {
  const { language, t, dir } = useLanguage();
  const { siteSettings, updateSiteSettings, exportData, importData } = useData();
  const [settings, setSettings] = useState(siteSettings);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleSave = () => {
    updateSiteSettings(settings);
    toast.success(t('تنظیمات ذخیره شد', 'Settings saved'));
  };

  const handleExport = () => {
    const data = exportData();
    const blob = new Blob([data], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `digital-shop-backup-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    toast.success(t('پشتیبان‌گیری انجام شد', 'Backup created'));
  };

  const handleImport = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (event) => {
      const content = event.target?.result as string;
      if (importData(content)) {
        toast.success(t('داده‌ها بازیابی شدند', 'Data restored'));
        window.location.reload();
      } else {
        toast.error(t('خطا در بازیابی داده‌ها', 'Error restoring data'));
      }
    };
    reader.readAsText(file);
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <h1 className="text-2xl md:text-3xl font-bold">{t('تنظیمات سایت', 'Site Settings')}</h1>
        <div className="flex gap-2">
          <Button variant="outline" onClick={handleExport}>
            <Download className="h-4 w-4 ml-2" />
            {t('پشتیبان‌گیری', 'Backup')}
          </Button>
          <Button variant="outline" onClick={() => fileInputRef.current?.click()}>
            <Upload className="h-4 w-4 ml-2" />
            {t('بازیابی', 'Restore')}
          </Button>
          <input
            ref={fileInputRef}
            type="file"
            accept=".json"
            onChange={handleImport}
            className="hidden"
          />
          <Button onClick={handleSave}>
            <Save className="h-4 w-4 ml-2" />
            {t('ذخیره', 'Save')}
          </Button>
        </div>
      </div>

      <Tabs defaultValue="general">
        <TabsList className="w-full justify-start">
          <TabsTrigger value="general">{t('عمومی', 'General')}</TabsTrigger>
          <TabsTrigger value="contact">{t('تماس', 'Contact')}</TabsTrigger>
          <TabsTrigger value="social">{t('شبکه‌های اجتماعی', 'Social')}</TabsTrigger>
          <TabsTrigger value="seo">{t('سئو', 'SEO')}</TabsTrigger>
        </TabsList>

        <TabsContent value="general" className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="siteName">{t('نام سایت (فارسی)', 'Site Name (FA)')}</Label>
              <Input
                id="siteName"
                value={settings.siteName}
                onChange={(e) => setSettings({ ...settings, siteName: e.target.value })}
                dir="rtl"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="siteNameEn">{t('نام سایت (انگلیسی)', 'Site Name (EN)')}</Label>
              <Input
                id="siteNameEn"
                value={settings.siteNameEn}
                onChange={(e) => setSettings({ ...settings, siteNameEn: e.target.value })}
                dir="ltr"
              />
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="logo">{t('URL لوگو', 'Logo URL')}</Label>
              <Input
                id="logo"
                value={settings.logo}
                onChange={(e) => setSettings({ ...settings, logo: e.target.value })}
                dir="ltr"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="favicon">{t('URL فاوآیکون', 'Favicon URL')}</Label>
              <Input
                id="favicon"
                value={settings.favicon}
                onChange={(e) => setSettings({ ...settings, favicon: e.target.value })}
                dir="ltr"
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="description">{t('توضیحات (فارسی)', 'Description (FA)')}</Label>
            <Input
              id="description"
              value={settings.description}
              onChange={(e) => setSettings({ ...settings, description: e.target.value })}
              dir="rtl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="descriptionEn">{t('توضیحات (انگلیسی)', 'Description (EN)')}</Label>
            <Input
              id="descriptionEn"
              value={settings.descriptionEn}
              onChange={(e) => setSettings({ ...settings, descriptionEn: e.target.value })}
              dir="ltr"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="footerText">{t('متن فوتر (فارسی)', 'Footer Text (FA)')}</Label>
            <Input
              id="footerText"
              value={settings.footerText}
              onChange={(e) => setSettings({ ...settings, footerText: e.target.value })}
              dir="rtl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="footerTextEn">{t('متن فوتر (انگلیسی)', 'Footer Text (EN)')}</Label>
            <Input
              id="footerTextEn"
              value={settings.footerTextEn}
              onChange={(e) => setSettings({ ...settings, footerTextEn: e.target.value })}
              dir="ltr"
            />
          </div>
        </TabsContent>

        <TabsContent value="contact" className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="phone">
              <Phone className="h-4 w-4 inline ml-2" />
              {t('تلفن', 'Phone')}
            </Label>
            <Input
              id="phone"
              value={settings.phone}
              onChange={(e) => setSettings({ ...settings, phone: e.target.value })}
              dir="ltr"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="email">
              <Globe className="h-4 w-4 inline ml-2" />
              {t('ایمیل', 'Email')}
            </Label>
            <Input
              id="email"
              type="email"
              value={settings.email}
              onChange={(e) => setSettings({ ...settings, email: e.target.value })}
              dir="ltr"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="address">{t('آدرس', 'Address')}</Label>
            <Input
              id="address"
              value={settings.address}
              onChange={(e) => setSettings({ ...settings, address: e.target.value })}
              dir="rtl"
            />
          </div>
        </TabsContent>

        <TabsContent value="social" className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="instagram">
              <Instagram className="h-4 w-4 inline ml-2" />
              Instagram
            </Label>
            <Input
              id="instagram"
              value={settings.socialLinks.instagram || ''}
              onChange={(e) =>
                setSettings({
                  ...settings,
                  socialLinks: { ...settings.socialLinks, instagram: e.target.value },
                })
              }
              dir="ltr"
              placeholder="https://instagram.com/..."
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="telegram">
              <Send className="h-4 w-4 inline ml-2" />
              Telegram
            </Label>
            <Input
              id="telegram"
              value={settings.socialLinks.telegram || ''}
              onChange={(e) =>
                setSettings({
                  ...settings,
                  socialLinks: { ...settings.socialLinks, telegram: e.target.value },
                })
              }
              dir="ltr"
              placeholder="https://t.me/..."
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="whatsapp">
              <Phone className="h-4 w-4 inline ml-2" />
              WhatsApp
            </Label>
            <Input
              id="whatsapp"
              value={settings.socialLinks.whatsapp || ''}
              onChange={(e) =>
                setSettings({
                  ...settings,
                  socialLinks: { ...settings.socialLinks, whatsapp: e.target.value },
                })
              }
              dir="ltr"
              placeholder="https://wa.me/..."
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="twitter">Twitter</Label>
            <Input
              id="twitter"
              value={settings.socialLinks.twitter || ''}
              onChange={(e) =>
                setSettings({
                  ...settings,
                  socialLinks: { ...settings.socialLinks, twitter: e.target.value },
                })
              }
              dir="ltr"
              placeholder="https://twitter.com/..."
            />
          </div>
        </TabsContent>

        <TabsContent value="seo" className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="keywords">{t('کلمات کلیدی', 'Keywords')}</Label>
            <Input
              id="keywords"
              value={settings.keywords}
              onChange={(e) => setSettings({ ...settings, keywords: e.target.value })}
              dir="rtl"
              placeholder={t('با کاما جدا کنید', 'Separate with commas')}
            />
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}
