import { Link } from 'react-router-dom';
import { Heart, ShoppingCart, Star } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { useLanguage } from '@/contexts/LanguageContext';
import { useCart } from '@/contexts/CartContext';
import type { Product } from '@/types';
import { toast } from 'sonner';

interface ProductCardProps {
  product: Product;
}

export function ProductCard({ product }: ProductCardProps) {
  const { language, t } = useLanguage();
  const { addToCart } = useCart();

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat(language === 'fa' ? 'fa-IR' : 'en-US').format(price);
  };

  const handleAddToCart = (e: React.MouseEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (product.stock > 0) {
      addToCart(product, 1);
    } else {
      toast.error(t('این محصول ناموجود است', 'This product is out of stock'));
    }
  };

  return (
    <div className="group relative bg-card rounded-xl border overflow-hidden hover:shadow-lg transition-all duration-300">
      {/* Badges */}
      <div className="absolute top-3 left-3 z-10 flex flex-col gap-2">
        {product.isNew && (
          <Badge className="bg-green-500 hover:bg-green-600">
            {t('جدید', 'New')}
          </Badge>
        )}
        {product.discount && product.discount > 0 && (
          <Badge variant="destructive">
            {product.discount}% {t('تخفیف', 'Off')}
          </Badge>
        )}
        {product.stock === 0 && (
          <Badge variant="secondary">
            {t('ناموجود', 'Out of Stock')}
          </Badge>
        )}
      </div>

      {/* Wishlist Button */}
      <button
        className="absolute top-3 right-3 z-10 w-8 h-8 bg-white/90 dark:bg-black/50 rounded-full flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity hover:bg-white"
        onClick={(e) => {
          e.preventDefault();
          e.stopPropagation();
          toast.success(t('به علاقه‌مندی‌ها اضافه شد', 'Added to favorites'));
        }}
      >
        <Heart className="h-4 w-4" />
      </button>

      {/* Image */}
      <Link to={`/product/${product.id}`}>
        <div className="aspect-square overflow-hidden bg-muted">
          <img
            src={product.images[0]}
            alt={language === 'fa' ? product.name : product.nameEn}
            className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
            loading="lazy"
          />
        </div>
      </Link>

      {/* Content */}
      <div className="p-4">
        <Link to={`/product/${product.id}`}>
          <h3 className="font-semibold text-sm md:text-base mb-2 line-clamp-2 hover:text-primary transition-colors">
            {language === 'fa' ? product.name : product.nameEn}
          </h3>
        </Link>

        {/* Rating */}
        <div className="flex items-center gap-1 mb-2">
          <Star className="h-4 w-4 fill-yellow-400 text-yellow-400" />
          <span className="text-sm font-medium">{product.rating}</span>
          <span className="text-xs text-muted-foreground">({product.reviewCount})</span>
        </div>

        {/* Price */}
        <div className="flex items-center gap-2 mb-3">
          <span className="font-bold text-lg text-primary">
            {formatPrice(product.price)} {t('تومان', 'Toman')}
          </span>
          {product.originalPrice && product.originalPrice > product.price && (
            <span className="text-sm text-muted-foreground line-through">
              {formatPrice(product.originalPrice)}
            </span>
          )}
        </div>

        {/* Add to Cart Button */}
        <Button
          onClick={handleAddToCart}
          disabled={product.stock === 0}
          className="w-full"
          size="sm"
        >
          <ShoppingCart className="h-4 w-4 ml-2" />
          {product.stock === 0 ? t('ناموجود', 'Out of Stock') : t('افزودن به سبد', 'Add to Cart')}
        </Button>
      </div>
    </div>
  );
}
